package com.agilex.healthcare.veteranappointment.restservice.directscheduling;

import com.agilex.healthcare.directscheduling.dataservice.CancelCodes;

import com.agilex.healthcare.directscheduling.domain.AppointmentTimeSlots;
import com.agilex.healthcare.directscheduling.domain.CancelReasons;
import com.agilex.healthcare.directscheduling.domain.VARBookedAppointmentCollections;
import com.agilex.healthcare.directscheduling.domain.PatientVisited;
import com.agilex.healthcare.directscheduling.domain.FacilityMemberTeam;
import com.agilex.healthcare.directscheduling.domain.PatientRequestLimit;
import com.agilex.healthcare.directscheduling.domain.BookedAppointment;
import com.agilex.healthcare.directscheduling.domain.BookedAppointmentCollection;

import com.agilex.healthcare.directscheduling.utils.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.veteranappointment.clientapi.JwtClient;
import com.agilex.healthcare.veteranappointment.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.veteranappointment.domain.DisabledFeatures;
import com.agilex.healthcare.veteranappointment.testutility.TestHelper;
import gov.va.vamf.scheduling.direct.domain.CdwClinic;
import gov.va.vamf.scheduling.direct.domain.CustomFriendlyText;
import gov.va.vamf.scheduling.direct.domain.TypesOfCare;
import gov.va.vamf.scheduling.direct.domain.AppointmentEligibilityAtInstitution;

import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.Ignore;

import javax.ws.rs.core.GenericType;
import javax.ws.rs.core.MediaType;

import java.io.IOException;
import java.util.List;
import java.util.Arrays;
import java.util.Collection;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;

public class DSResourceTest {

	private static String devBaseServerUrl = "http://localhost:9092/var/VeteranAppointmentRequestService/v4/rest";
	private static String institutionsByTypeOfCareTestUrl = "/direct-scheduling/institutions";
	private static String appointmentSlotsTestUrl = "/direct-scheduling/site/523/patient/ICN/1006088937/available-appointment-slots";
	private static String cancelReasonTestUrl = "/direct-scheduling/site/523/patient/ICN/1006088937/cancel-reasons-list";
	private static String allPendingAppointmentsForPatientTestUrl = "/direct-scheduling/site/523/patient/ICN/1006088937/booked-appointments";
	private static String typesOfCareTestUrl = "/clinical-services/types-of-care";
	private static String clinicsByTypeOfCareTestUrl = "/clinical-services/clinics";
	private static String patientVisitedInPastMonthsDirectByTypeOfCareTestUrl = "/direct-scheduling/site/523/patient/ICN/1006088937/direct-eligibility/visited-in-past-months";
	private static String patientVisitedInPastMonthsRequestByTypeOfCareTestUrl = "/direct-scheduling/site/523/patient/ICN/1006088937/request-eligibility/visited-in-past-months";
	private static String pactTeamTestUrl = "/direct-scheduling/site/523/patient/ICN/1006088937/pact-team";
	private static String patientRequestLimitTestUrl = "/direct-scheduling/patient/ICN/1006088937/request-limit";
	private static String customFriendlyTextTestUrl = "/direct-scheduling/custom-friendly-text";
	private static String providersTestUrl = "/direct-scheduling/site/523/patient/ICN/1006088937/providers";
	private static String teamListTestUrl = "/direct-scheduling/site/523/patient/ICN/1006088937/team-list";
	private static String disabledFeaturesTestUrl = "/disabled-features";

	private static MobileHealthClientTestVersion client;

	private static String[] typesOfCare = {"203", "323", "408", "502", "CR1"};
	private static String[] typesOfCareInstitutionPairs = {"323:523", "203:523", "408:523", "502:523"};
	private static String CLINIC_IDS = "488";
	private static String TH_FEATURE_SET = "TH_FEATURE_SET";
	private static String JWT_HEADER_NAME = "x-vamf-jwt";

	private static JwtClient jwtClient = new JwtClient();
	private static String jwt;

	@BeforeClass
	public static void fixtureSetup() throws IOException{
		jwt = jwtClient.getJwt();
		client = TestHelper.createMobileHealthClient();
	}

	@AfterClass
	public static void tearDown() {
		if (client != null) {
			client.getJerseyClient().close();
			client = null;
		}
	}

	// Allocate time slots for specific clinic in VistA.
	@Test
	public void getAppointmentSlots() {
		Collection<AppointmentTimeSlots> appointmentTimeSlotsC =
				client.getJerseyClient().target(devBaseServerUrl)
						.path(appointmentSlotsTestUrl)
						.queryParam("startDate", DateHelper.formatDate(DateHelper.getTomorrow()))
						.queryParam("endDate", DateHelper.formatDate(DateHelper.get120DaysFromNow()))
						.queryParam("clinicIds", "430")
						.request(MediaType.APPLICATION_JSON)
						.accept(MediaType.APPLICATION_JSON)
						.header(JWT_HEADER_NAME, jwt)
						.get(new GenericType<Collection<AppointmentTimeSlots>>() {
						});

		assertNotNull("No appointment time slots found for clinic: 430.", appointmentTimeSlotsC);

		for (AppointmentTimeSlots appointmentTimeSlots : appointmentTimeSlotsC) {

			assertNotNull(appointmentTimeSlots);
			assertTrue("Check for only primary care slots for clinic: 430.", appointmentTimeSlots.getClinicId().equals("430"));
			assertTrue("Check to make sure some slots are returned.", appointmentTimeSlots.getSize() >= 1);
		}
	}

	// Data should already exist in VistA.
	@Test
	public void getCancelReasonsDropDownList() {
		CancelReasons cancelReasons = client.getJerseyClient().target(devBaseServerUrl)
				.path(cancelReasonTestUrl)
				.request(MediaType.APPLICATION_JSON)
				.accept(MediaType.APPLICATION_JSON)
				.header(JWT_HEADER_NAME, jwt)
				.get(CancelReasons.class);

		assertNotNull(cancelReasons);
		assertTrue("Check to make sure cancel reasons are returned.", cancelReasons.getSize() >= 1);
	}

	// Create appointment through application.
	@Test
	public void getAllPendingAppointmentsForPatient() {
		VARBookedAppointmentCollections varBookedAppointmentCollections = client.getJerseyClient().target(devBaseServerUrl)
				.path(allPendingAppointmentsForPatientTestUrl)
				.queryParam("startDate", DateHelper.formatDate(DateHelper.getYesterday()))
				.request(MediaType.APPLICATION_JSON)
				.accept(MediaType.APPLICATION_JSON)
				.header(JWT_HEADER_NAME, jwt)
				.get(VARBookedAppointmentCollections.class);

		assertNotNull("No booked appointment collections found.", varBookedAppointmentCollections);

		boolean foundPendingAppointment = false;

		for (BookedAppointmentCollection bookedAppointmentCollection : varBookedAppointmentCollections.getBookedAppointmentCollections()) {

			assertNotNull("No booked appointment collection found.", bookedAppointmentCollection);

			Collection<BookedAppointment> bookedAppointments = bookedAppointmentCollection.getBookedAppointments();

			if (foundPendingAppointment = hasBookedAppointment(bookedAppointments)) {
				break;
			}
		}

		assertTrue("No pending appointments found for patient.", foundPendingAppointment);
	}

	// Create primary care appointment through application.
	@Test
	public void getPrimaryCarePendingAppointmentsForPatient() {
		VARBookedAppointmentCollections varBookedAppointmentCollections = client.getJerseyClient().target(devBaseServerUrl)
				.path(allPendingAppointmentsForPatientTestUrl)
				.queryParam("startDate", DateHelper.formatDate(DateHelper.getYesterday()))
				.queryParam("clinicIds", CLINIC_IDS)
				.request(MediaType.APPLICATION_JSON)
				.accept(MediaType.APPLICATION_JSON)
				.header(JWT_HEADER_NAME, jwt)
				.get(VARBookedAppointmentCollections.class);

		assertNotNull("No booked appointment collections found.", varBookedAppointmentCollections);

		boolean foundAppointmentClinic = false;

		for (BookedAppointmentCollection bookedAppointmentCollection : varBookedAppointmentCollections.getBookedAppointmentCollections()) {

			assertNotNull("No booked appointment collection found.", bookedAppointmentCollection);

			if (!(bookedAppointmentCollection.getCollectionName().equalsIgnoreCase("PRIMARY_CARE"))) {
				continue;
			}

			Collection<BookedAppointment> bookedAppointments = bookedAppointmentCollection.getBookedAppointments();

			assertTrue("No booked appointments found.", !bookedAppointments.isEmpty());


			if (foundAppointmentClinic = hasBookedAppointmentClinic(bookedAppointments, Arrays.asList(CLINIC_IDS.split(",")))) {
				break;
			}

		}

		assertTrue("No appointments found for clinics " + CLINIC_IDS, foundAppointmentClinic);
	}

	// Create appointments through application.
	@Test
	public void getAllAppointmentsClinicFriendlyNameTest() {
		VARBookedAppointmentCollections bookedAppointmentsCollections = client.getJerseyClient().target(devBaseServerUrl)
				.path(allPendingAppointmentsForPatientTestUrl)
				.queryParam("startDate", DateHelper.formatDate(DateHelper.getYesterday()))
				.request(MediaType.APPLICATION_JSON)
				.accept(MediaType.APPLICATION_JSON)
				.header(JWT_HEADER_NAME, jwt)
				.get(VARBookedAppointmentCollections.class);

		assertTrue("No booked appointment collections found.", bookedAppointmentsCollections != null
				&& !bookedAppointmentsCollections.getBookedAppointmentCollections().isEmpty());

		boolean foundClinicFriendlyName = false;

		for (BookedAppointmentCollection bookedAppointmentCollection : bookedAppointmentsCollections.getBookedAppointmentCollections()) {
			Collection<BookedAppointment> bookedAppointments = bookedAppointmentCollection.getBookedAppointments();

			assertFalse("No booked appointments found.", bookedAppointmentCollection.getBookedAppointments().isEmpty());

			for (BookedAppointment bookedAppointment : bookedAppointments) {

				assertNotNull("Booked appointment has empty clinic.", bookedAppointment.getClinic());

				if (!CancelCodes.hasCancelCode(bookedAppointment.getCurrentStatus()) &&
						NullChecker.isNotNullish(bookedAppointment.getClinic().getFriendlyName())) {
					foundClinicFriendlyName = true;

					break;
				}
			}
		}

		assertTrue("No clinic friendly name found for a booked appointment.", foundClinicFriendlyName);
	}

	// Types of care should already exist in Mongo clinical services collection.
	@Test
	public void getTypesOfCare() {
		List<TypesOfCare> typesOfCares =
				client.getJerseyClient().target(devBaseServerUrl)
						.path(typesOfCareTestUrl)
						.request(MediaType.APPLICATION_JSON)
						.accept(MediaType.APPLICATION_JSON)
						.header(JWT_HEADER_NAME, jwt)
						.get(new GenericType<List<TypesOfCare>>() {
						});

		assertNotNull("No types of care found.", typesOfCares);
	}

	// Set up Mongo direct and/or request eligibility criteria collections for specific institution by type of care.
	@Test
	public void getEligibleInstitutionsByTypeOfCare() {
		for (int i = 0; i < typesOfCare.length; i++) {
			Collection<AppointmentEligibilityAtInstitution> appointmentEligibilityAtInstitutions =
					client.getJerseyClient().target(devBaseServerUrl)
							.path(institutionsByTypeOfCareTestUrl)
							.queryParam("facility-code", "523")
							.queryParam("clinical-service", typesOfCare[i])
							.request(MediaType.APPLICATION_JSON)
							.accept(MediaType.APPLICATION_JSON)
							.header(JWT_HEADER_NAME, jwt)
							.get(new GenericType<Collection<AppointmentEligibilityAtInstitution>>() {
							});

			assertTrue("No eligible institutions for type of care: " + typesOfCare[i],
					appointmentEligibilityAtInstitutions != null && !appointmentEligibilityAtInstitutions.isEmpty());
		}
	}

	// Data should already exist in CDW.
	@Test
	public void getClinicsByTypeOfCare() {
		for (int i = 0; i < typesOfCareInstitutionPairs.length; i++) {
			String[] codePairs = typesOfCareInstitutionPairs[i].split(":");

			Collection<CdwClinic> clinics =
					client.getJerseyClient().target(devBaseServerUrl)
							.path(clinicsByTypeOfCareTestUrl)
							.queryParam("clinical-service", codePairs[0])
							.queryParam("institution-code", codePairs[1])
							.request(MediaType.APPLICATION_JSON)
							.accept(MediaType.APPLICATION_JSON)
							.header(JWT_HEADER_NAME, jwt)
							.get(new GenericType<Collection<CdwClinic>>() {
							});

			assertTrue("No clinics for type of care: " + codePairs[0] + " institution: " + codePairs[1],
					clinics != null && !clinics.isEmpty());
		}
	}

	// NOTE: Ignoring. Needs booked appointment with CHECKED IN or CHECHED OUT status,
	// which does not exist in development environment.
	@Ignore
	@Test
	public void getVisitedInPastMonthsDirectByTypeOfCare() {
		PatientVisited visited =
				client.getJerseyClient().target(devBaseServerUrl)
						.path(patientVisitedInPastMonthsDirectByTypeOfCareTestUrl)
						.queryParam("clinical-service", "203")
						.queryParam("institution-code", "523BZ")
						.request(MediaType.APPLICATION_JSON)
						.accept(MediaType.APPLICATION_JSON)
						.header(JWT_HEADER_NAME, jwt)
						.get(PatientVisited.class);

		assertTrue("Patient has not visited a clinic for direct eligibility in the past " + visited.getDurationInMonths() + " months for type of care Audiology", visited.getDurationInMonths() != 0 && visited.getHasVisitedInPastMonths());
	}

	// NOTE: Ignoring. Needs booked appointment with CHECKED IN or CHECHED OUT status,
	// which does not exist in development environment.
	@Ignore
	@Test
	public void getVisitedInPastMonthsRequestByTypeOfCare() {
		PatientVisited visited =
				client.getJerseyClient().target(devBaseServerUrl)
						.path(patientVisitedInPastMonthsRequestByTypeOfCareTestUrl)
						.queryParam("clinical-service", "408")
						.queryParam("institution-code", "523DT")
						.request(MediaType.APPLICATION_JSON)
						.accept(MediaType.APPLICATION_JSON)
						.header(JWT_HEADER_NAME, jwt)
						.get(PatientVisited.class);

		assertTrue("Patient has not visited a clinic for request eligibility in the past " + visited.getDurationInMonths() + " months for type of care Optometry", visited.getDurationInMonths() != 0 && visited.getHasVisitedInPastMonths());
	}

	// Data should already exist in CDW.
	@Test
	public void getPactTeam() {
		Collection<FacilityMemberTeam> pactTeamMembers =
				client.getJerseyClient().target(devBaseServerUrl)
						.path(pactTeamTestUrl)
						.request(MediaType.APPLICATION_JSON)
						.accept(MediaType.APPLICATION_JSON)
						.header(JWT_HEADER_NAME, jwt)
						.get(new GenericType<Collection<FacilityMemberTeam>>() {
						});

		assertTrue("No pact team members for patient.", pactTeamMembers != null
				&& !pactTeamMembers.isEmpty());
	}

	// Set up Mongo data for request eligibility criteria submitted request limit on specific institution and type of care
	@Test
	public void getPatientMetRequestLimit() {
		PatientRequestLimit patientRequestLimit =
				client.getJerseyClient().target(devBaseServerUrl)
						.path(patientRequestLimitTestUrl)
						.queryParam("clinical-service", "408")
						.queryParam("institution-code", "523")
						.request(MediaType.APPLICATION_JSON)
						.accept(MediaType.APPLICATION_JSON)
						.header(JWT_HEADER_NAME, jwt)
						.get(PatientRequestLimit.class);

		assertTrue("Requests exceeds limit for patient.", patientRequestLimit.getNumberOfRequests() < patientRequestLimit.getRequestLimit());
	}

	// Set up Mongo data in custom friendly text collection for specific institution.
	@Test
	public void getCustomFriendlyText() {
		CustomFriendlyText customFriendlyText =
				client.getJerseyClient().target(devBaseServerUrl)
						.path(customFriendlyTextTestUrl)
						.queryParam("institution-code", "523")
						.request(MediaType.APPLICATION_JSON)
						.accept(MediaType.APPLICATION_JSON)
						.header(JWT_HEADER_NAME, jwt)
						.get(CustomFriendlyText.class);

		assertTrue("No custom friendly text found for institution 523BZ", customFriendlyText != null
				&& !customFriendlyText.getFriendlyText().isEmpty());
	}

	// Data should already exist in CDW.
	@Test
	public void getProviders() {
		Collection<FacilityMemberTeam> providers =
				client.getJerseyClient().target(devBaseServerUrl)
						.path(providersTestUrl)
						.queryParam("clinical-service", "323")
						.request(MediaType.APPLICATION_JSON)
						.accept(MediaType.APPLICATION_JSON)
						.header(JWT_HEADER_NAME, jwt)
						.get(new GenericType<Collection<FacilityMemberTeam>>() {
						});

		assertTrue("No providers for patient.", providers != null
				&& !providers.isEmpty());
	}

	// Data should already exist in CDW.
	@Test
	public void getTeamList() {
		Collection<FacilityMemberTeam> teamList =
				client.getJerseyClient().target(devBaseServerUrl)
						.path(teamListTestUrl)
						.queryParam("clinical-service", "323")
						.request(MediaType.APPLICATION_JSON)
						.accept(MediaType.APPLICATION_JSON)
						.header(JWT_HEADER_NAME, jwt)
						.get(new GenericType<Collection<FacilityMemberTeam>>() {
						});

		assertTrue("No team list for patient.", teamList != null
				&& !teamList.isEmpty());

	}


	// Add TH_FEATURE_SET in var-disabled-features.properties
	@Test
	public void getTelehealthDisabledFeature() {
		DisabledFeatures disabledFeatures =
				client.getJerseyClient().target(devBaseServerUrl)
						.path(disabledFeaturesTestUrl)
						.request(MediaType.APPLICATION_JSON)
						.accept(MediaType.APPLICATION_JSON)
						.header(JWT_HEADER_NAME, jwt)
						.get(DisabledFeatures.class);


		assertTrue("No TH_FEATURE_SET found in disabled features.", disabledFeatures.getDisabledFeatures().contains(TH_FEATURE_SET));
	}

	public boolean hasBookedAppointment(Collection<BookedAppointment> bookedAppointments) {
		for (BookedAppointment bookedAppointment : bookedAppointments) {
			if (bookedAppointment.getCurrentStatus() != null &&
					!CancelCodes.hasCancelCode(bookedAppointment.getCurrentStatus())) {
				return true;
			}
		}

		return false;
	}

	public boolean hasBookedAppointmentClinic(Collection<BookedAppointment> bookedAppointments, Collection<String> clinicIds) {

		boolean hasClinic = false;

		for (BookedAppointment bookedAppointment : bookedAppointments) {
			if (bookedAppointment.getClinic().getId() != null && clinicIds.contains(bookedAppointment.getClinic().getId())) {
				hasClinic = true;
			}
		}

		return hasClinic;
	}
}
